<?php

/*
 * IconCaptcha - Copyright 2023, Fabian Wennink (https://www.fabianwennink.nl)
 * Licensed under the MIT license: https://www.fabianwennink.nl/projects/IconCaptcha/license
 *
 * The above copyright notice and license shall be included in all copies or substantial portions of the software.
 */

namespace IconCaptcha\Challenge\Hooks;

use IconCaptcha\Session\SessionInterface;

class Hook
{
    /**
     * Attempts to call the given hook and return the value generated by the hook.
     * In case the given hook does not exist, the given default value will be returned.
     *
     * @param string $type The name of the hook in the options.
     * @param mixed $class The interface which the hook has to implement in order to be called properly.
     * @param string $action The name of the function defined in the interface, which should be called.
     * @param SessionInterface $session The session containing captcha information.
     * @param array $options The captcha options.
     * @param mixed $default The default value, which will be returned if the hook could not be called.
     * @param mixed ...$params Any additional data which has to be passed to the hook.
     * @return mixed The result of the hook, or the default value if no hook was called.
     * @throws InvalidHookException If an attempt was made to call a hook, but failed as it was configured incorrectly.
     */
    public static function call(string $type, $class, string $action, SessionInterface $session, array $options, $default, ...$params)
    {
        // Get the hook.
        $hook = self::getHook($options, $type, $class);

        // Ensure the hook is valid and has a callable action.
        if (!empty($hook) && is_callable([$hook, $action], true)) {
            return $hook->{$action}($_REQUEST, $session, $options, ...array_values($params));
        }

        return $default;
    }

    /**
     * Attempts to call the given hook.
     *
     * @param string $type The name of the hook in the options.
     * @param mixed $class The interface which the hook has to implement in order to be called properly.
     * @param string $action The name of the function defined in the interface, which should be called.
     * @param SessionInterface $session The session containing captcha information.
     * @param array $options The captcha options.
     * @param mixed ...$params Any additional data which has to be passed to the hook.
     * @throws InvalidHookException If an attempt was made to call a hook, but failed as it was configured incorrectly.
     */
    public static function callVoid(string $type, $class, string $action, SessionInterface $session, array $options, ...$params): void
    {
        // Get the hook.
        $hook = self::getHook($options, $type, $class);

        // Ensure the hook is valid and has a callable action.
        if (!empty($hook) && is_callable([$hook, $action], true)) {
            $hook->{$action}($_REQUEST, $session, $options, ...array_values($params));
        }
    }

    /**
     * Attempts to return a class instance of the hook based on the given hook name.
     *
     * @param array $options The captcha options.
     * @param string $hookName The name of the hook in the options.
     * @param mixed $interface The interface which the hook has to implement in order to be called properly.
     * @return mixed|null The hook class instance, or NULL if no hook was defined for the current action.
     * @throws InvalidHookException If an attempt was made to call a hook, but failed as it was configured incorrectly.
     */
    private static function getHook(array $options, string $hookName, $interface)
    {
        // Make sure the hook exists in the options.
        if (!isset($options['hooks'][$hookName])) {
            return null;
        }

        $option = $options['hooks'][$hookName];

        // Make sure the hook is a valid class before invoking it.
        if (is_string($option) && class_exists($option)) {
            $hook = new $option();
            if (in_array($interface, class_implements($hook), true)) {
                return $hook;
            }
        }

        // The hook is invalid, and most likely incorrectly configured.
        throw new InvalidHookException($hookName);
    }
}
